// Background Service Worker
console.log('AI Studio 导出插件 Background Service Worker 启动');

// 后台服务配置
const SERVICE_URL = 'http://115.190.230.181:3005';

// 监听来自content script的消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('Background收到消息:', request);

  if (request.action === 'exportAsImage') {
    handleExportAsImage(request.data, sender, sendResponse);
    return true; // 保持消息通道开放
  } else if (request.action === 'exportAsPDF') {
    handleExportAsPDF(request.data, sender, sendResponse);
    return true;
  }
});

// 处理导出为图片
async function handleExportAsImage(messageData, sender, sendResponse) {
  const startTime = Date.now();
  try {
    console.log('开始导出为图片...');
    console.log('HTML内容大小:', messageData.html.length, '字符');

    // 构建完整的HTML
    const fullHTML = buildFullHTML(messageData);

    // 发送到后台服务
    const response = await fetch(`${SERVICE_URL}/api/convert/image`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        html: fullHTML,
        options: {
          format: 'png',
          // PNG不支持quality参数，移除
          backgroundColor: '#ffffff'
        }
      })
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error('服务器错误响应:', errorText);
      throw new Error(`服务器响应错误: ${response.status}`);
    }

    // 获取图片数据
    const arrayBuffer = await response.arrayBuffer();

    // 转换为base64 data URL
    const base64 = btoa(
      new Uint8Array(arrayBuffer).reduce((data, byte) => data + String.fromCharCode(byte), '')
    );
    const dataUrl = `data:image/png;base64,${base64}`;

    // 生成文件名
    const filename = `aistudio-export-${Date.now()}.png`;

    // 使用data URL下载（Service Worker兼容）
    chrome.downloads.download({
      url: dataUrl,
      filename: filename,
      saveAs: true
    }, (downloadId) => {
      if (chrome.runtime.lastError) {
        console.error('下载失败:', chrome.runtime.lastError);
        sendResponse({ success: false, error: chrome.runtime.lastError.message });
      } else {
        const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
        console.log(`下载成功: ${downloadId} (耗时 ${elapsed}秒)`);
        sendResponse({
          success: true,
          message: `图片已保存 (${elapsed}秒)`
        });
      }
    });

  } catch (error) {
    const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
    console.error(`导出图片失败 (${elapsed}秒):`, error);
    sendResponse({ success: false, error: error.message });
  }
}

// 处理导出为PDF
async function handleExportAsPDF(messageData, sender, sendResponse) {
  const startTime = Date.now();
  try {
    console.log('开始导出为PDF...');
    console.log('HTML内容大小:', messageData.html.length, '字符');

    // 构建完整的HTML
    const fullHTML = buildFullHTML(messageData);

    // 发送到后台服务
    const response = await fetch(`${SERVICE_URL}/api/convert/pdf`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        html: fullHTML,
        options: {
          format: 'A4',
          printBackground: true,
          margin: {
            top: '20px',
            right: '20px',
            bottom: '20px',
            left: '20px'
          }
        }
      })
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error('服务器错误响应:', errorText);
      throw new Error(`服务器响应错误: ${response.status}`);
    }

    // 获取PDF数据
    const arrayBuffer = await response.arrayBuffer();

    // 转换为base64 data URL
    const base64 = btoa(
      new Uint8Array(arrayBuffer).reduce((data, byte) => data + String.fromCharCode(byte), '')
    );
    const dataUrl = `data:application/pdf;base64,${base64}`;

    // 生成文件名
    const filename = `aistudio-export-${Date.now()}.pdf`;

    // 使用data URL下载（Service Worker兼容）
    chrome.downloads.download({
      url: dataUrl,
      filename: filename,
      saveAs: true
    }, (downloadId) => {
      if (chrome.runtime.lastError) {
        console.error('下载失败:', chrome.runtime.lastError);
        sendResponse({ success: false, error: chrome.runtime.lastError.message });
      } else {
        const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
        console.log(`下载成功: ${downloadId} (耗时 ${elapsed}秒)`);
        sendResponse({
          success: true,
          message: `PDF已保存 (${elapsed}秒)`
        });
      }
    });

  } catch (error) {
    const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
    console.error(`导出PDF失败 (${elapsed}秒):`, error);
    sendResponse({ success: false, error: error.message });
  }
}

// 构建完整的HTML文档
function buildFullHTML(messageData) {
  return `
<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>AI Studio 导出</title>
  <style>
    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
    }

    body {
      font-family: 'Google Sans', 'Roboto', Arial, sans-serif;
      font-size: 14px;
      line-height: 1.6;
      color: #202124;
      background: #ffffff;
      padding: 20px;
    }

    .export-container {
      max-width: 800px;
      margin: 0 auto;
      background: white;
    }

    .export-header {
      margin-bottom: 20px;
      padding-bottom: 10px;
      border-bottom: 2px solid #e8eaed;
    }

    .export-header h1 {
      font-size: 20px;
      font-weight: 500;
      color: #1967d2;
    }

    .export-header .timestamp {
      font-size: 12px;
      color: #5f6368;
      margin-top: 5px;
    }

    .export-content {
      padding: 10px 0;
    }

    /* 代码块样式 */
    pre {
      background: #f8f9fa;
      border: 1px solid #e8eaed;
      border-radius: 8px;
      padding: 16px;
      overflow-x: auto;
      margin: 16px 0;
    }

    code {
      font-family: 'Roboto Mono', monospace;
      font-size: 13px;
    }

    /* 标题样式 */
    h1, h2, h3, h4, h5, h6 {
      margin: 16px 0 8px;
      font-weight: 500;
    }

    h1 { font-size: 24px; }
    h2 { font-size: 20px; }
    h3 { font-size: 18px; }

    /* 段落样式 */
    p {
      margin: 8px 0;
    }

    /* 列表样式 */
    ul, ol {
      margin: 8px 0;
      padding-left: 24px;
    }

    li {
      margin: 4px 0;
    }

    /* 表格样式 */
    table {
      border-collapse: collapse;
      width: 100%;
      margin: 16px 0;
    }

    th, td {
      border: 1px solid #e8eaed;
      padding: 8px 12px;
      text-align: left;
    }

    th {
      background: #f8f9fa;
      font-weight: 500;
    }

    /* Material Icons字体 */
    .material-symbols-outlined {
      font-family: 'Material Symbols Outlined';
      font-weight: normal;
      font-style: normal;
      font-size: 24px;
      display: inline-block;
      line-height: 1;
      text-transform: none;
      letter-spacing: normal;
      word-wrap: normal;
      white-space: nowrap;
      direction: ltr;
    }

    ${messageData.styles || ''}
  </style>
</head>
<body>
  <div class="export-container">
    <div class="export-header">
      <h1>AI Studio 对话导出</h1>
      <div class="timestamp">导出时间: ${new Date(messageData.timestamp).toLocaleString('zh-CN')}</div>
    </div>
    <div class="export-content">
      ${messageData.html}
    </div>
  </div>
</body>
</html>
  `.trim();
}

// 插件安装时的处理
chrome.runtime.onInstalled.addListener((details) => {
  console.log('插件已安装/更新:', details);

  if (details.reason === 'install') {
    console.log('首次安装插件');
    // 可以打开欢迎页面或设置页面
  } else if (details.reason === 'update') {
    console.log('插件已更新到版本:', chrome.runtime.getManifest().version);
  }
});
