// Content Script - 注入到AI Studio页面
// 增强的调试版本

// 调试日志函数
function debugLog(message, data = null) {
  const timestamp = new Date().toLocaleTimeString();
  console.log(`[AI Studio Export] ${timestamp} - ${message}`, data || '');
}

debugLog('🚀 插件内容脚本开始加载...');

// 检查是否在正确的页面
if (!window.location.hostname.includes('aistudio.google.com')) {
  debugLog('❌ 不在 AI Studio 页面，插件不会激活');
} else {
  debugLog('✓ 在 AI Studio 页面，插件已激活');
}

// 菜单项配置
const MENU_ITEMS = [
  {
    id: 'save-as-image',
    icon: 'image',
    label: '保存为图片',
    action: 'exportAsImage'
  },
  {
    id: 'save-as-pdf',
    icon: 'picture_as_pdf',
    label: '保存为PDF',
    action: 'exportAsPDF'
  }
];

// 当前正在操作的消息元素
let currentMessageElement = null;
let menuObserver = null;

// 初始化插件
function init() {
  debugLog('📋 初始化插件...');

  // 等待页面完全加载
  if (document.readyState === 'loading') {
    debugLog('⏳ 等待DOM加载完成...');
    document.addEventListener('DOMContentLoaded', () => {
      debugLog('✓ DOM已加载');
      startPlugin();
    });
  } else {
    debugLog('✓ DOM已就绪');
    startPlugin();
  }
}

function startPlugin() {
  // 检查页面元素
  checkPageElements();

  // 使用MutationObserver监听DOM变化
  observeMenuChanges();

  // 监听来自background script的消息
  chrome.runtime.onMessage.addListener(handleMessage);

  debugLog('✓ 插件初始化完成');
}

// 检查页面关键元素
function checkPageElements() {
  debugLog('🔍 检查页面元素...');

  const messages = document.querySelectorAll('ms-chat-turn');
  debugLog(`找到 ${messages.length} 个对话消息`);

  const menus = document.querySelectorAll('.mat-mdc-menu-content');
  debugLog(`找到 ${menus.length} 个菜单`);

  const moreButtons = document.querySelectorAll('ms-chat-turn-options button');
  debugLog(`找到 ${moreButtons.length} 个更多选项按钮`);
}

// 监听菜单的打开 - 使用更简单的方法
function observeMenuChanges() {
  debugLog('👁️ 开始监听页面变化...');

  // 监听整个文档的变化
  menuObserver = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      mutation.addedNodes.forEach((node) => {
        if (node.nodeType === 1) {
          // 检查是否是菜单面板
          if (node.classList && (
              node.classList.contains('cdk-overlay-pane') ||
              node.classList.contains('mat-mdc-menu-panel')
          )) {
            debugLog('🎯 检测到菜单打开！', node);
            const menu = node.querySelector('.mat-mdc-menu-content');
            if (menu) {
              debugLog('✓ 找到菜单内容', menu);
              handleMenuOpen(menu, node);
            }
          }

          // 检查节点内部是否有菜单
          const menu = node.querySelector('.mat-mdc-menu-content');
          if (menu) {
            debugLog('✓ 在新节点内找到菜单', menu);
            handleMenuOpen(menu, node);
          }
        }
      });
    });
  });

  menuObserver.observe(document.body, {
    childList: true,
    subtree: true
  });

  debugLog('✓ 监听器已设置');
}

// 处理菜单打开
function handleMenuOpen(menu, overlayPane) {
  debugLog('📂 处理菜单打开事件');

  // 找到对应的消息元素
  findCurrentMessage();

  // 添加自定义菜单项
  addCustomMenuItems(menu);
}

// 找到当前操作的消息元素
function findCurrentMessage() {
  debugLog('🔍 查找当前消息元素...');

  const allMessages = document.querySelectorAll('ms-chat-turn');
  debugLog(`页面共有 ${allMessages.length} 个消息`);

  // 找到最后一个model消息（AI回复）
  for (let i = allMessages.length - 1; i >= 0; i--) {
    const msg = allMessages[i];
    const content = msg.querySelector('.turn-content');
    if (content) {
      currentMessageElement = msg;
      debugLog('✓ 找到当前消息', {
        index: i,
        hasContent: true
      });
      return;
    }
  }

  debugLog('⚠️ 未找到合适的消息元素');
}

// 添加自定义菜单项
function addCustomMenuItems(menuContent) {
  debugLog('➕ 添加自定义菜单项...');

  // 检查是否已添加过（避免重复）
  if (menuContent.querySelector('#save-as-image')) {
    debugLog('⚠️ 菜单项已存在，跳过');
    return;
  }

  // 添加分隔线（可选，如果想要分隔线的话）
  // const separator = document.createElement('div');
  // separator.style.cssText = 'height: 1px; background: #e0e0e0; margin: 8px 0;';
  // menuContent.appendChild(separator);

  MENU_ITEMS.forEach(item => {
    const menuItem = createMenuItem(item);
    menuContent.appendChild(menuItem);
    debugLog(`✓ 添加菜单项: ${item.label}`);
  });

  debugLog('✓ 所有菜单项添加完成');
}

// 创建菜单项元素
function createMenuItem(config) {
  const button = document.createElement('button');
  button.id = config.id;
  button.className = 'mat-mdc-menu-item mat-focus-indicator ng-star-inserted aistudio-export-item';
  button.setAttribute('mat-menu-item', '');
  button.setAttribute('role', 'menuitem');
  button.setAttribute('tabindex', '0');
  button.setAttribute('aria-disabled', 'false');

  // 不添加额外样式，保持和原生菜单一致
  button.innerHTML = `
    <span class="mat-mdc-menu-item-text">
      <span aria-hidden="true" class="material-symbols-outlined notranslate start-icon">
        ${config.icon}
      </span>
      <span>${config.label}</span>
    </span>
    <div matripple="" class="mat-ripple mat-mdc-menu-ripple"></div>
  `;

  button.addEventListener('click', (e) => {
    debugLog(`🖱️ 点击菜单项: ${config.label}`);
    e.preventDefault();
    e.stopPropagation();
    handleMenuAction(config.action);
  });

  return button;
}

// 处理菜单动作
async function handleMenuAction(action) {
  debugLog(`⚡ 执行动作: ${action}`);

  if (!currentMessageElement) {
    debugLog('❌ 未找到消息内容');
    showNotification('错误', '未找到消息内容', 'error');
    return;
  }

  try {
    // 显示加载提示
    const loadingNotification = showLoadingNotification(action);

    // 提取消息内容
    debugLog('📦 开始提取消息内容...');
    updateLoadingNotification(loadingNotification, '正在提取内容...');

    const messageData = extractMessageContent(currentMessageElement);
    debugLog('✓ 消息内容提取完成', {
      htmlLength: messageData.html.length,
      stylesLength: messageData.styles.length
    });

    // 发送到background script处理
    debugLog('📡 发送消息到后台脚本...');
    updateLoadingNotification(loadingNotification, '正在生成文件...');

    chrome.runtime.sendMessage({
      action: action,
      data: messageData
    }, (response) => {
      // 移除加载提示
      hideLoadingNotification(loadingNotification);

      if (chrome.runtime.lastError) {
        debugLog('❌ 后台脚本通信错误', chrome.runtime.lastError);
        showNotification('错误', chrome.runtime.lastError.message, 'error');
        return;
      }

      debugLog('✓ 收到后台响应', response);

      if (response && response.success) {
        showNotification('成功', response.message || '导出成功', 'success');
      } else {
        showNotification('错误', response?.error || '导出失败', 'error');
      }
    });

  } catch (error) {
    debugLog('❌ 导出错误', error);
    console.error('导出错误详情:', error);
    showNotification('错误', error.message, 'error');
  }
}

// 提取消息内容
function extractMessageContent(messageElement) {
  debugLog('📄 提取消息内容...');

  const contentElement = messageElement.querySelector('.turn-content');

  if (!contentElement) {
    throw new Error('未找到消息内容元素');
  }

  // 获取HTML内容
  const htmlContent = contentElement.innerHTML;
  debugLog(`✓ HTML内容长度: ${htmlContent.length} 字符`);

  // 获取计算后的样式
  const styles = extractStyles(contentElement);
  debugLog(`✓ 样式长度: ${styles.length} 字符`);

  return {
    html: htmlContent,
    styles: styles,
    timestamp: new Date().toISOString()
  };
}

// 提取样式 - 简化版本
function extractStyles(element) {
  const styles = [];

  // 获取内联样式
  const inlineStyles = Array.from(document.querySelectorAll('style'))
    .map(style => style.textContent)
    .join('\n');

  styles.push(inlineStyles);

  // 基本样式
  styles.push(`
    * { box-sizing: border-box; }
    body { font-family: 'Google Sans', 'Roboto', Arial, sans-serif; }
  `);

  return styles.join('\n');
}

// 显示加载通知
function showLoadingNotification(action) {
  debugLog('⏳ 显示加载通知');

  // 移除旧的通知
  const oldNotifications = document.querySelectorAll('.aistudio-export-notification, .aistudio-export-loading');
  oldNotifications.forEach(n => n.remove());

  const actionName = action === 'exportAsImage' ? '图片' : 'PDF';

  const loading = document.createElement('div');
  loading.className = 'aistudio-export-loading';
  loading.innerHTML = `
    <div style="
      position: fixed;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%);
      background: rgba(0, 0, 0, 0.85);
      color: white;
      padding: 30px 40px;
      border-radius: 12px;
      z-index: 10001;
      text-align: center;
      box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
      min-width: 300px;
    ">
      <div class="loading-spinner" style="
        width: 48px;
        height: 48px;
        border: 4px solid rgba(255, 255, 255, 0.3);
        border-top-color: white;
        border-radius: 50%;
        animation: spin 0.8s linear infinite;
        margin: 0 auto 20px;
      "></div>
      <div style="font-size: 18px; font-weight: 500; margin-bottom: 8px;">
        正在导出${actionName}
      </div>
      <div class="loading-message" style="font-size: 14px; opacity: 0.8;">
        正在准备...
      </div>
    </div>
  `;

  // 添加旋转动画
  if (!document.querySelector('#loading-spinner-animation')) {
    const style = document.createElement('style');
    style.id = 'loading-spinner-animation';
    style.textContent = `
      @keyframes spin {
        to { transform: rotate(360deg); }
      }
    `;
    document.head.appendChild(style);
  }

  document.body.appendChild(loading);

  // 添加淡入效果
  setTimeout(() => {
    loading.style.opacity = '1';
    loading.style.transition = 'opacity 0.2s ease';
  }, 10);

  return loading;
}

// 更新加载通知的消息
function updateLoadingNotification(loadingElement, message) {
  if (loadingElement) {
    const messageEl = loadingElement.querySelector('.loading-message');
    if (messageEl) {
      messageEl.textContent = message;
    }
  }
}

// 隐藏加载通知
function hideLoadingNotification(loadingElement) {
  if (loadingElement) {
    loadingElement.style.opacity = '0';
    setTimeout(() => {
      loadingElement.remove();
    }, 200);
  }
}

// 显示通知 - 增强版
function showNotification(title, message, type = 'info') {
  debugLog(`📢 显示通知: ${title} - ${message}`);

  // 移除旧通知（但不移除加载通知）
  const oldNotification = document.querySelector('.aistudio-export-notification');
  if (oldNotification) {
    oldNotification.remove();
  }

  const notification = document.createElement('div');
  notification.className = 'aistudio-export-notification';

  const colors = {
    success: '#2e7d32',
    error: '#c62828',
    info: '#1565c0'
  };

  const icons = {
    success: '✓',
    error: '✗',
    info: 'ℹ'
  };

  notification.innerHTML = `
    <div style="
      position: fixed;
      top: 20px;
      right: 20px;
      background: white;
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      padding: 16px 20px;
      z-index: 10000;
      min-width: 300px;
      max-width: 400px;
      border-left: 4px solid ${colors[type]};
      animation: slideIn 0.3s ease;
    ">
      <div style="display: flex; align-items: center; gap: 12px;">
        <div style="
          width: 32px;
          height: 32px;
          border-radius: 50%;
          background: ${colors[type]};
          color: white;
          display: flex;
          align-items: center;
          justify-content: center;
          font-weight: bold;
          font-size: 18px;
        ">${icons[type]}</div>
        <div style="flex: 1;">
          <div style="font-weight: 500; color: #202124; margin-bottom: 4px;">${title}</div>
          <div style="font-size: 13px; color: #5f6368;">${message}</div>
        </div>
      </div>
    </div>
  `;

  // 添加动画样式
  if (!document.querySelector('#aistudio-export-animations')) {
    const style = document.createElement('style');
    style.id = 'aistudio-export-animations';
    style.textContent = `
      @keyframes slideIn {
        from { opacity: 0; transform: translateX(100px); }
        to { opacity: 1; transform: translateX(0); }
      }
    `;
    document.head.appendChild(style);
  }

  document.body.appendChild(notification);

  setTimeout(() => {
    notification.style.opacity = '0';
    notification.style.transform = 'translateX(100px)';
    notification.style.transition = 'all 0.3s ease';
    setTimeout(() => {
      notification.remove();
    }, 300);
  }, 3000);
}

// 处理来自background的消息
function handleMessage(request, sender, sendResponse) {
  debugLog('📨 收到来自background的消息', request);

  if (request.action === 'ping') {
    sendResponse({ status: 'ok', message: '插件运行正常' });
  }

  return true;
}

// 页面加载完成后初始化
init();

// 添加全局测试函数
window.aiStudioExportDebug = {
  checkElements: checkPageElements,
  testNotification: () => {
    showNotification('测试', '这是一条测试通知', 'success');
  },
  showLog: () => {
    console.log('当前消息元素:', currentMessageElement);
    console.log('菜单观察器:', menuObserver);
  }
};

debugLog('✓ 插件加载完成！可以在控制台使用 aiStudioExportDebug 进行调试');
